function data = msh_piles(data,execute_mode)

    S=     data.Soil;
    PG=    data.PileGroup;
    W=     data.Frequencies;
    Mopt=  data.MeshOptions;

    % Initialize
    PileSurfaceIntersects = false(PG.N,S.N);
    PileSurfaceX= zeros(PG.N,S.N,3);
    % Determine intersection between piles and surfaces
    for kl = 1:S.N
        for kp = 1:PG.N
            % Find local coordinate of the intersection between the pile
            % and surface. Pile head is at t=0 and tip at t=1.
            z = S.layer(kl).z;
            head = PG.Pile(kp).xhead;
            tip = PG.Pile(kp).xtip;
            t = (z-head(3))/(tip(3)-head(3));
            if t>=0 && t<=1
                PileSurfaceX(kp,kl,1) = (1-t)*head(1) + t*tip(1);
                PileSurfaceX(kp,kl,2) = (1-t)*head(2) + t*tip(2);
                PileSurfaceX(kp,kl,3) = z;
                PileSurfaceIntersects(kp,kl) = true;
            end
        end
    end
    % Build each pile head and tip on each layer
    PileLayerPresent = false(PG.N,S.N);
    PileLayerHeadX= zeros(PG.N,S.N,3);
    PileLayerTipX= zeros(PG.N,S.N,3);
    for kl = 1:S.N
        for kp = 1:PG.N
            if PileSurfaceIntersects(kp,kl) && PG.Pile(kp).xtip(3)<S.layer(kl).z
                if S.bedrock && kl == S.N
                    f_display_error(execute_mode,'Invalid configuration, a pile is inside the bedrock');
                else
                    PileLayerPresent(kp,kl) = true;
                    PileLayerHeadX(kp,kl,:) = PileSurfaceX(kp,kl,:);
                    if kl < S.N
                        if PG.Pile(kp).xtip(3)<S.layer(kl+1).z
                            PileLayerTipX(kp,kl,:) = PileSurfaceX(kp,kl+1,:);
                        else
                            PileLayerTipX(kp,kl,:) = PG.Pile(kp).xtip;
                        end
                    else
                        PileLayerTipX(kp,kl,:) = PG.Pile(kp).xtip;
                    end
                end
            end
        end
    end

    NPiles=  PG.N;
    NLayers= S.N;
    PilesD= zeros(PG.N,1);
    for idpil=1:PG.N
        PilesD(idpil)= PG.Pile(idpil).Do;
    end
    LayerCS= zeros(S.N,1);
    for isoil=1:S.N
        LayerCS(isoil)= S.layer(isoil).cs;
    end
    switch W.type
        case 1
            switch W.ref
                case 1
                    maxf = max(W.value)*W.ref_data(2)/W.ref_data(1)/(2*pi);
                case 2
                    maxf = max(W.value)*W.ref_data(1)/(2*pi);
                case 3
                    maxf = max(W.value)*W.ref_data(1);
                otherwise
                    f_display_error(execute_mode,'Invalid type of normalization')
            end
        case 2
            maxf = max(W.value)/(2*pi);
        case 3
            maxf = max(W.value);
        otherwise
            f_display_error(execute_mode,'Invalid type of frequency')
    end

    % Find the number of elements according to pile mesh parameters
    PileLayerNElements = zeros(NPiles,NLayers);
    for kl = 1:NLayers
        for kp = 1:NPiles
          if PileLayerPresent(kp,kl)
              h = PileLayerHeadX(kp,kl,:);
              t = PileLayerTipX(kp,kl,:);
              L = sqrt(dot(h-t,h-t));
              D = PilesD(kp);
              if L<=D
                  f_display_warning(execute_mode,'Pile diameter is greater than layer thickness');
                  PileLayerNElements(kp,kl) = 1;
              else
                  lambdaS = LayerCS(kl)/maxf;
                  lmin = D/Mopt.nepd_pile;
                  lmax = lambdaS/Mopt.nelo_pile;
                  if lmax<lmin
                      lmax = lmin;
                  end
                  l = (1-Mopt.t)*lmin+Mopt.t*lmax;
                  PileLayerNElements(kp,kl) = ceil(L/l);
                  if L/PileLayerNElements(kp,kl)<D && PileLayerNElements(kp,kl)>1
                      PileLayerNElements(kp,kl) = PileLayerNElements(kp,kl)-1;
                  end
              end
          end
        end
    end

    % Generate pile mesh for each layer
    nxp   = [];
    enp   = [];
    ennp  = [];
    edp   = [];
    pidx  = []; % pile index
    llidx = []; % line load index
    lld   = []; % domain or layer of the line load
    klp=0;
    for kl = 1:NLayers
        for kp = 1:NPiles
            if PileLayerPresent(kp,kl)
                klp = klp + 1;
                x1 = reshape(PileLayerTipX(kp,kl,:),3,1);
                x2 = reshape(PileLayerHeadX(kp,kl,:),3,1);
                [n,e] = SubdivideLine2(x1,x2,PileLayerNElements(kp,kl));
                [n,e,nn] = TransformToQuadratic(n',e',2*ones(length(e(1,:)),1),ones(length(e(1,:)),1));
                % Renumber nodes according to z
                [~,idx] = sort(n(:,3),'ascend');
                n = n(idx,:);
                ne = e;
                for kn=1:length(n(:,1))
                    [row,col]=find(e==idx(kn));
                    for kfnd=1:length(row)
                        ne(row(kfnd),col(kfnd)) = kn;
                    end
                end
                e = ne;
                if ~isempty(nxp)
                    nnodesprev = length(nxp(:,1));
                else
                    nnodesprev = 0;
                end                
                nxp   = [nxp  ;n];
                enp   = [enp  ;e+nnodesprev];
                ennp  = [ennp ;nn];
                edp   = [edp  ;ones(length(e(:,1)),1)];
                pidx  = [pidx ;kp*ones(length(e(:,1)),1)];
                llidx = [llidx;klp*ones(length(e(:,1)),1)];
                lld   = [lld  ;kl*ones(length(e(:,1)),1)];
            end
        end
    end
    pn = nxp;
    pe = [pidx lld llidx edp ennp enp(:,[1:3]) zeros(length(lld),6)];

    % Save 
    data.Mesh.pn = pn;
    data.Mesh.pe = pe;

    disp('Line mesh of piles at each layer created ...')
end

function f_display_error(execute_mode,msg)
    switch execute_mode
        case 1 % GUI
            errordlg(msg, 'Error', 'modal')
        case 2 % command 
            error(msg)
        otherwise
            error('Wrong execution mode selected.')
    end
    return
end
function f_display_warning(execute_mode,msg)
    switch execute_mode
        case 1 % GUI
            warndlg(msg, 'Alert', 'modal')
        case 2 % command 
            warning(msg)
        otherwise
            error('Wrong execution mode selected.')
    end
    return
end
