function data= msh_Gmsh_S4(data)
    % 
    % Writes down the mesh file for GMSH program and plots the mesh.
    %
    % Symmetry: 1 (no symmetry), 2 (half-symmetry), 4 (quarter-symmetry)

    %% INPUT

    % http://stackoverflow.com/questions/8340818/start-application-from-matlab

    % Save library paths
    MatlabPath = getenv('LD_LIBRARY_PATH');

    % Make Matlab use system libraries
    setenv('LD_LIBRARY_PATH',getenv('PATH'));

    C =    data.PileGroupType;
    PG =   data.PileGroup;
    S =    data.Soil;
    W =    data.Frequencies;
    Mopt = data.MeshOptions;

    for id = 1:PG.N
        Do(id) = PG.Pile(id).Do;
    end

    % Origin of coordinates
    x0 = 0;
    y0 = 0;


    % Initialize
    PileSurfaceIntersects = false(PG.N,S.N);
    PileSurfaceX = zeros(PG.N,S.N,3);
    % Determine intersection between piles and surfaces
    for kl = 1:S.N
        for kp = 1:PG.N
            % Find local coordinate of the intersection between the pile
            % and surface. Pile head is at t=0 and tip at t=1.
            z = S.layer(kl).z;
            head = PG.Pile(kp).xhead;
            tip = PG.Pile(kp).xtip;
            t = (z-head(3))/(tip(3)-head(3));
            if t>=0 && t<=1
                PileSurfaceX(kp,kl,1) = (1-t)*head(1) + t*tip(1);
                PileSurfaceX(kp,kl,2) = (1-t)*head(2) + t*tip(2);
                PileSurfaceX(kp,kl,3) = z;
                PileSurfaceIntersects(kp,kl) = true;
            end
        end
    end
    % Build each pile head and tip on each layer
    PG.PileLayerPresent = false(PG.N,S.N);
    PG.PileLayerHeadX = zeros(PG.N,S.N,3);
    PG.PileLayerTipX = zeros(PG.N,S.N,3);
    for kl = 1:S.N
        z = S.layer(kl).z;
        for kp = 1:PG.N
            if PileSurfaceIntersects(kp,kl) && PG.Pile(kp).xtip(3)<S.layer(kl).z
                if S.bedrock && kl == S.N
                    errordlg('A pile is inside the bedrock','Error','modal');
                    error ('Invalid configuration, a pile is inside the bedrock');
                else
                    PG.PileLayerPresent(kp,kl) = true;
                    PG.PileLayerHeadX(kp,kl,:) = PileSurfaceX(kp,kl,:);
                    if kl < S.N
                        if PG.Pile(kp).xtip(3)<S.layer(kl+1).z
                            PG.PileLayerTipX(kp,kl,:) = PileSurfaceX(kp,kl+1,:);
                        else
                            PG.PileLayerTipX(kp,kl,:) = PG.Pile(kp).xtip;
                        end
                    else
                        PG.PileLayerTipX(kp,kl,:) = PG.Pile(kp).xtip;
                    end
                end
            end
        end
    end

    %data.PileGroup = PG;


    % %  Limits of the mesh elements' size  % %

    switch W.type
        case 1
            switch W.ref
                case 1
                    freq_f = max(W.value)*W.ref_data(2)/W.ref_data(1)/(2*pi);
                case 2
                    freq_f = max(W.value)*W.ref_data(1)/(2*pi);
                case 3
                    freq_f = max(W.value)*W.ref_data(1);
                otherwise
                    error ('Invalid type of normalization')
            end
        case 2
            freq_f = max(W.value)/(2*pi);
        case 3
            freq_f = max(W.value);
        otherwise
            error ('Invalid type of frequency')
    end

    ms_near = zeros(S.N,1);
    ms_far =  zeros(S.N,1);

    for iddom = 1:S.N
        if iddom==1
            ms_near(iddom) = (S.layer(iddom).cs/freq_f) / Mopt.nelo_near;
            ms_far(iddom) =  (S.layer(iddom).cs/freq_f) / Mopt.nelo_far;
        else
            if iddom==S.N && S.bedrock
                ms_near(iddom) = (S.layer(iddom-1).cs/freq_f) / Mopt.nelo_near;
                ms_far(iddom) =  (S.layer(iddom-1).cs/freq_f) / Mopt.nelo_far;
            else
                cs = min(S.layer(iddom-1).cs,S.layer(iddom).cs);
                ms_near(iddom) = (cs/freq_f) / Mopt.nelo_near;
                ms_far(iddom) =  (cs/freq_f) / Mopt.nelo_far;
            end
        end

        % Maximum foundation dimension is taken from the bounding box
        Points = [PG.Pile.xhead;PG.Pile.xtip];
        MaxFoundDimX = 2*(max(Points(:,1)) - min(Points(:,1)));
        MaxFoundDimY = 2*(max(Points(:,2)) - min(Points(:,2)));
        MaxFoundDimZ =    max(Points(:,3)) - min(Points(:,3)) ;
        MaxFoundDim =  max([MaxFoundDimX MaxFoundDimY MaxFoundDimZ]);

        R = MaxFoundDim * Mopt.ratio_R;

        ms_near(iddom) = min(ms_near(iddom),max(Do)/Mopt.nepl_near);
        ms_far(iddom) =  min(ms_far(iddom), R/Mopt.nepl_far);
    end

    fprintf('\n');

    geofile = fullfile(strcat(data.Path,'feGMSH.geo')); 
    OUP =     fopen(geofile,'w');

    fprintf(OUP,'// Maximum pile length:\n');
    fprintf(OUP,'L=%5.3f;\n\n',max(C.l));

    fprintf(OUP,'// Foundation halfwidth (x and y axis):\n');
    fprintf(OUP,'b_x=%5.3f;\n'  ,C.bx);
    fprintf(OUP,'b_y=%5.3f;\n\n',C.by);

    fprintf(OUP,'//Origin of coordinates (x and y axis):\n');
    fprintf(OUP,'x0=%7.1f;\n',  x0);
    fprintf(OUP,'y0=%7.1f;\n\n',y0);


    for iddom = 1:S.N

        % Some variables more to make the file writing easier...

        ndom = (iddom-1)*100;
        z =    S.layer(iddom).z;

        % Writing mesh parameters

        if iddom==S.N && S.bedrock
            x_max = max(PG.PileLayerHeadX(:,iddom-1,1));
            y_max = max(PG.PileLayerHeadX(:,iddom-1,2));

            ms_near(iddom) =   ms_near(iddom-1);
            ms_far(iddom) =    ms_far(iddom-1);

            ms_inside =   sprintf('ms_inside_id_%d',iddom-1);
            msfar =       sprintf('ms_far_id_%d',iddom-1);
            Ltruncint =   sprintf('Ltruncint_id_%d',iddom-1);
            Ltruncext =   sprintf('Ltruncext_id_%d',iddom-1);
            Ltruncclose = sprintf('Ltruncclose_id_%d',iddom-1);
            nd_close =    sprintf('nd_close_id_%d',iddom-1);
        else
            x_max = max(PG.PileLayerHeadX(:,iddom,1));
            y_max = max(PG.PileLayerHeadX(:,iddom,2));

            ms_inside =   sprintf('ms_inside_id_%d',iddom);
            msfar =       sprintf('ms_far_id_%d',iddom);
            Ltruncint =   sprintf('Ltruncint_id_%d',iddom);
            Ltruncext =   sprintf('Ltruncext_id_%d',iddom);
            Ltruncclose = sprintf('Ltruncclose_id_%d',iddom);
            nd_close =    sprintf('nd_close_id_%d',iddom);
        end

        l = max(x_max,y_max);
        Lint = max( l*Mopt.ratio_r , l + max(Do)*Mopt.ratio_D );
        Lext = Mopt.ratio_R*MaxFoundDim;

        fprintf(OUP,'// Mesh parameters\n');
        fprintf(OUP,'ms_max_id_%d =  %f;\n',iddom,ms_far(iddom));
        fprintf(OUP,'ms_far_id_%d =  ms_max_id_%d;\n',iddom,iddom);
        fprintf(OUP,'ms_inside_id_%d =  %f;\n',iddom,ms_near(iddom));
        fprintf(OUP,'Ltruncint_id_%d =  %f;\n',iddom,Lint);
        fprintf(OUP,'Ltruncext_id_%d =    %f;\n',iddom,Lext);
        fprintf(OUP,'Ltruncclose_id_%d =  Ltruncext_id_%d + ms_far_id_%d;\n',iddom,iddom,iddom);


        fprintf(OUP,'\n');

        fprintf(OUP,'/*\n');
        if iddom==1
            fprintf(OUP,'Free-surface\n');
        elseif iddom==S.N && S.bedrock
            fprintf(OUP,'Bedrock\n');
        else
            fprintf(OUP,'Interface between stratum %d and stratum %d\n',    ...
                                                                iddom-1,iddom);
        end
        fprintf(OUP,'*/');   fprintf(OUP,'\n');


        % Points
        fprintf(OUP,'Point (%d) = {x0,y0,%f,%s};\n',       1+ndom,                        z,ms_inside);
        fprintf(OUP,'Point (%d) = {%s,y0,%f,%s};\n',       2+ndom,Ltruncint  ,            z,ms_inside);
        fprintf(OUP,'Point (%d) = {%s,%s,%f,%s};\n',       3+ndom,Ltruncint  ,Ltruncint  ,z,ms_inside);
        fprintf(OUP,'Point (%d) = {x0,y0+%s,%f,%s};\n',    4+ndom,Ltruncint  ,            z,ms_inside);
        fprintf(OUP,'Point (%d) = {x0+%s,y0,%f,%s};\n',    5+ndom,Ltruncext  ,            z,msfar);
        fprintf(OUP,'Point (%d) = {x0+%s,y0+%s,%f,%s};\n', 6+ndom,Ltruncext  ,Ltruncext,  z,msfar);
        fprintf(OUP,'Point (%d) = {x0,y0+%s,%f,%s};\n',    7+ndom,Ltruncext  ,            z,msfar);
        fprintf(OUP,'Point (%d) = {x0+%s,y0,%f,%s};\n',    8+ndom,Ltruncclose,            z,msfar);
        fprintf(OUP,'Point (%d) = {x0+%s,y0+%s,%f,%s};\n', 9+ndom,Ltruncclose,Ltruncext,  z,msfar);
        fprintf(OUP,'Point (%d) = {x0+%s,y0+%s,%f,%s};\n',10+ndom,Ltruncclose,Ltruncclose,z,msfar);
        fprintf(OUP,'Point (%d) = {x0+%s,y0+%s,%f,%s};\n',11+ndom,Ltruncext  ,Ltruncclose,z,msfar);
        fprintf(OUP,'Point (%d) = {x0,y0+%s,%f,%s};\n',   12+ndom,Ltruncclose,            z,msfar);

        % Lines

        % % Inner area

        % points in inner area
        x_points = [];
        y_points = [];

        for idpil = 1:PG.N
            if PG.PileLayerPresent(idpil,iddom)
                if iddom==S.N && S.bedrock
                    x_layer = PG.PileLayerHeadX(idpil,iddom-1,1);
                    y_layer = PG.PileLayerHeadX(idpil,iddom-1,2);
                else
                    x_layer = PG.PileLayerHeadX(idpil,iddom,1);
                    y_layer = PG.PileLayerHeadX(idpil,iddom,2);
                end
            else
                if PG.PileLayerPresent(idpil,iddom-1) && PG.PileLayerHeadX(idpil,iddom-1,3)==z
                    x_layer = PG.PileLayerTipX(idpil,iddom-1,1);
                    y_layer = PG.PileLayerTipX(idpil,iddom-1,2);
                end
            end

            if isempty(find(x_points==x_layer))
                x_points = [x_points x_layer];
            end

            if isempty(find(y_points==y_layer))
                y_points = [y_points y_layer];
            end
        end

        if isempty(find(x_points==x0))
            x_points = [x0 x_points];
        end
        if isempty(find(y_points==y0))
            y_points = [y0 y_points];
        end

        if isempty(find(x_points==(Lint + x0)))
            x_points = [x_points (Lint + x0)];
        end
        if isempty(find(y_points==(Lint + y0)))
            y_points = [y_points (Lint + y0)];
        end

        x_points = sort(x_points);
        y_points = sort(y_points);

        id =            12+ndom;
        points_index =  zeros(length(x_points),length(y_points));

        for i_y = 1:length(y_points)
            for i_x = 1:length(x_points)

                x = x_points(i_x);
                y = y_points(i_y);

                % Check if coordinates are the same of another point
                if x==x0
                    px0 = true;
                else
                    px0 = false;
                end

                if x==(x0+Lint)
                    pxL = true;
                else
                    pxL = false;
                end

                if y==y0
                    py0 = true;
                else
                    py0 = false;
                end

                if y==(y0+Lint)
                    pyL = true;
                else
                    pyL = false;
                end

                % 
                if     (px0 && py0);    points_index(i_x,i_y) = 1 + ndom;
                elseif (pxL && py0);    points_index(i_x,i_y) = 2 + ndom;
                elseif (pxL && pyL);    points_index(i_x,i_y) = 3 + ndom;
                elseif (px0 && pyL);    points_index(i_x,i_y) = 4 + ndom;
                else
                    ms = ms_near(iddom);
                    id = id + 1;
                    points_index(i_x,i_y) = id;

                    fprintf(OUP,'Point (%d) = {%f,%f,%f,%f};\n',id,x,y,z,ms);
                end
            end
        end

        % lines in inner area
        l = 20 + ndom;
        px0_indexes = [];
        py0_indexes = [];
        pxL_indexes = [];
        pyL_indexes = [];

        for i_y = 1:length(y_points)
            for i_x = 1:length(x_points)

                x = x_points(i_x);
                y = y_points(i_y);

                % Check if coordinates are the same of another point
                if x==x0
                    px0 = true;
                else
                    px0 = false;
                end
                if x==(x0+Lint)
                    pxL = true;
                else
                    pxL = false;
                end
                if y==y0
                    py0 = true;
                else
                    py0 = false;
                end
                if y==(y0+Lint)
                    pyL = true;
                else
                    pyL = false;
                end

                p1 = points_index(i_x,i_y);

                if px0 && py0
                    p2 = points_index(i_x+1,i_y  );
                    p3 = points_index(i_x  ,i_y+1);

                    l = l + 1;
                    py0_indexes = [py0_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p2);

                    l = l + 1;
                    px0_indexes = [px0_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p3);

                elseif py0 && ~px0 && i_x~=length(x_points)
                    p2 = points_index(i_x+1,i_y  );

                    l = l + 1;
                    py0_indexes = [py0_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p2);

                elseif px0 && ~py0 && i_y~=length(y_points)
                    p3 = points_index(i_x  ,i_y+1);

                    l = l + 1;
                    px0_indexes = [px0_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p3);

                elseif pxL && i_y~=length(y_points)
                    p3 = points_index(i_x  ,i_y+1);

                    l = l + 1;
                    pxL_indexes = [pxL_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p3);

                elseif pyL && i_x~=length(x_points)
                    p2 = points_index(i_x+1,i_y  );

                    l= l + 1;
                    pyL_indexes = [pyL_indexes l];
                    fprintf(OUP,'Line (%d) = {%d,%d};\n',l,p1,p2);

                end
            end
        end

        % Line loop of inner area
        l = l + 1;
        fprintf(OUP,'Line Loop (%d) = { %d ',l,py0_indexes(1));

        if length(py0_indexes) == 2
            fprintf(OUP,',%d',py0_indexes(2));
        elseif length(py0_indexes) > 2
            for id_l=2:length(py0_indexes)
                fprintf(OUP,',%d',py0_indexes(id_l));
            end
        end

        for id_l = pxL_indexes
            fprintf(OUP,',%d', id_l);
        end
        for id_l = pyL_indexes
            fprintf(OUP,',%d',-id_l);
        end
        for id_l = px0_indexes
            fprintf(OUP,',%d',-id_l);
        end

        fprintf(OUP,' };\n');

        fprintf(OUP,'Plane Surface (%d) = {%d};\n',1+ndom,l);

        % Points in inner area that dont define the lines
        for i_x = 1:length(x_points)
            for i_y = 1:length(y_points)
                x = x_points(i_x);
                y = y_points(i_y);

                % Check if coordinates are the same of another point
                if x==x0
                    px0 = true;
                else
                    px0 = false;
                end
                if x==(x0+Lint)
                    pxL = true;
                else
                    pxL = false;
                end
                if y==y0
                    py0 = true;
                else
                    py0 = false;
                end
                if y==(y0+Lint)
                    pyL = true;
                else
                    pyL = false;
                end

                if ~(px0 || pxL || py0 || pyL)
                    id = points_index(i_x,i_y);

                    fprintf(OUP,'Point{%d} In Surface{%d};\n',id,1+ndom);
                end
            end
        end


        % % Far area, x axis
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+1,2+ndom,5+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+2,5+ndom,6+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+3,6+ndom,3+ndom);

        fprintf(OUP,'Line Loop (%d) = {%d,%d,%d',4+l,1+l,2+l,3+l);
        for id_l = pxL_indexes
            fprintf(OUP,',%d',-id_l);
        end
        fprintf(OUP,' };\n');

        fprintf(OUP,'Plane Surface (%d) = {%d};\n',      2+ndom,4+l);

        % % Far area, y axis
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+5,6+ndom,7+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+6,7+ndom,4+ndom);

        fprintf(OUP,'Line Loop (%d) = {%d,%d,%d',7+l,(-1)*(3+l),5+l,6+l);
        for id_l = pyL_indexes
            fprintf(OUP,',%d',id_l);
        end
        fprintf(OUP,' };\n');

        fprintf(OUP,'Plane Surface (%d) = {%d};\n',      3+ndom,7+l);


        % % Closing area, x axis
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+8,5+ndom,8+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+9,8+ndom,9+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+10,9+ndom,6+ndom);
        fprintf(OUP,'Transfinite Line {%d,%d} = %s;\n',8+l,10+l,'1');
        fprintf(OUP,'Line Loop (%d) = {%d,%d,%d,%d};\n',11+l,8+l,9+l,10+l,(-1)*(2+l));
        fprintf(OUP,'Plane Surface (%d) = {%d};\n',4+ndom,11+l);
        fprintf(OUP,'Recombine Surface {%d};\n',   4+ndom);
        fprintf(OUP,'Transfinite Surface {%d} = {%d,%d,%d,%d};\n',4+ndom,5+ndom,8+ndom,9+ndom,6+ndom);

        % % Closing area, square
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+12, 9+ndom,10+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+13,10+ndom,11+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+14,11+ndom, 6+ndom);
        fprintf(OUP,'Transfinite Line {%d,%d,%d} = %s;\n',12+l,13+l,14+l,'1');
        fprintf(OUP,'Line Loop (%d) = {%d,%d,%d,%d};\n',15+l,12+l,13+l,14+l,(-1)*(10+l));
        fprintf(OUP,'Plane Surface (%d) = {%d};\n',5+ndom,15+l);
        fprintf(OUP,'Recombine Surface {%d};\n',   5+ndom);
        fprintf(OUP,'Transfinite Surface {%d} = {%d,%d,%d,%d};\n',5+ndom,9+ndom,10+ndom,11+ndom,6+ndom);

        % % Closing area, y axis
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+16,11+ndom,12+ndom);
        fprintf(OUP,'Line (%d) = {%d,%d};\n',l+17,12+ndom, 7+ndom);
        fprintf(OUP,'Transfinite Line {%d} = %s;\n',17+l, '1');
        fprintf(OUP,'Line Loop (%d) = {%d,%d,%d,%d};\n',18+l,16+l,17+l,(-1)*(5+l),(-1)*(14+l));
        fprintf(OUP,'Plane Surface (%d) = {%d};\n',6+ndom,18+l);
        fprintf(OUP,'Recombine Surface {%d};\n',6+ndom);
        fprintf(OUP,'Transfinite Surface {%d} = {%d,%d,%d,%d};\n',6+ndom,11+ndom,12+ndom,7+ndom,6+ndom);


        % 'physical surface'
        if iddom==1
            fprintf(OUP,'Physical Surface (''Free-surface'') = {%d,%d,%d};\n',1+ndom,2+ndom,3+ndom);
        else
            if iddom==S.N && S.bedrock
                fprintf(OUP,'Physical Surface (''Bedrock'') = {%d,%d,%d};\n',1+ndom,2+ndom,3+ndom);
            else
                fprintf(OUP,'Physical Surface (''Interface_%d-%d'') = {%d,%d,%d};\n',iddom-1,iddom,1+ndom,2+ndom,3+ndom);
            end
        end

        % Closing physical surface
        if iddom==1
            fprintf(OUP,'Physical Surface (''Free-surface-Close'') = {%d,%d,%d};\n',4+ndom,5+ndom,6+ndom);
        else
            if iddom==S.N && S.bedrock
                fprintf(OUP,'Physical Surface (''Bedrock-Close'') = {%d,%d,%d};\n',4+ndom,5+ndom,6+ndom);
            else
                fprintf(OUP,'Physical Surface (''Interface_%d-%d-Close'') = {%d,%d,%d};\n',iddom-1,iddom,4+ndom,5+ndom,6+ndom);
            end
        end

        fprintf(OUP,'\n');


    end % for S.N
    fprintf('\n\n');

    mshfile = fullfile(data.Path,'feGMSH.msh');

    % feGMSH%d.geo
    fprintf(OUP,'Mesh.Algorithm = 6;\n');
    fprintf(OUP,'Mesh.SurfaceFaces = 1;\n');
    fprintf(OUP,'Mesh.ColorCarousel = 2;\n');
    fprintf(OUP,'SetOrder 2;\n');
    fclose(OUP);

    % feGMSH%d.geo.opt
    meshoptfile = strcat(geofile,'.opt');

    OUP = fopen(meshoptfile,'w');
    fprintf(OUP,'Mesh.Algorithm = 6;\n');
    fprintf(OUP,'Mesh.SurfaceFaces = 1;\n');
    fprintf(OUP,'Mesh.ColorCarousel = 2;\n');
    fclose(OUP);


    % Mesh generation using GMSH
    if isunix || ismac
        meshing = sprintf('%s -2 -order 2 %s -o %s',Mopt.toolpath,geofile,mshfile);
    elseif ispc
        meshing = sprintf('%s -2 -order 2 %s -o %s',strcat(Mopt.toolpath,'.exe'),geofile,mshfile);
    end
    system(meshing);
    fprintf('\n\n');
    [status,~] = system(meshing,'-echo');
    if status~=0
        disp(status)
        error('Some error ocurred trying to run Gmsh');
    else
        fprintf(sprintf('Gmsh has ended running %s\n',mshfile));
    end


    % Reading data from file...

    INPmsh = fopen(mshfile,'r+');

    for nl = 1:4
        line = fgets(INPmsh);
    end
    nlmax = fscanf(INPmsh,'%d',1);

    for nl = 1:(nlmax+3)
        line = fgets(INPmsh);
    end
    num_sn = fscanf(INPmsh,'%d',1);

    % nodes' identifier and coordinates
    sn = zeros(num_sn,3);
    for cont_n = 1:num_sn
        nodeid =       fscanf(INPmsh,'%d',1);

        sn(cont_n,1) = fscanf(INPmsh,'%f',1);
        sn(cont_n,2) = fscanf(INPmsh,'%f',1);
        sn(cont_n,3) = fscanf(INPmsh,'%f',1);
    end

    for nl = 1:3
        line = fgets(INPmsh);
    end
    num_se = fscanf(INPmsh,'%d',1);

    e_dim =         2*ones(num_se,1);
    e_nodes_num =   zeros(num_se,1);
    gmsh_elements = zeros(num_se,14);

    for cont_e = 1:num_se
        for cont_n = 1:5
            % element nodes
            gmsh_elements(cont_e,cont_n) = fscanf(INPmsh,'%d',1);

            if cont_n==2
                switch gmsh_elements(cont_e,cont_n)
                    case 9    % triangular element
                        e_nodes_num(cont_e) = 6;
                        gmsh_elements(cont_e,cont_n) = 6;
                    case 10   % quadrangular element
                        e_nodes_num(cont_e) = 9;
                        gmsh_elements(cont_e,cont_n) = 9;
                end
            end
        end

        for cont_n = 6:(e_nodes_num(cont_e)+5)
            gmsh_elements(cont_e,cont_n) = fscanf(INPmsh,'%d',1);
        end
    end

    fclose(INPmsh);

    % ordered according to the defined surfaces (not the physical ones)
    ordered_elements = sortrows(gmsh_elements,5);
    e_surf =           ordered_elements(:,4);

    e_nodes_num =     zeros(num_se,1);
    e_nodes_indexes = zeros(num_se,9);

    for cont_e = 1:num_se
        e_nodes_num(cont_e) = length(find(ordered_elements(cont_e,6:14)));

        element_nodes = ordered_elements(cont_e,(6:e_nodes_num(cont_e)+5));

        if e_nodes_num(cont_e)==9
            e_nodes_indexes(cont_e,:) = element_nodes;
        else
            extra_zeros = zeros(1,9 - e_nodes_num(cont_e));
            e_nodes_indexes(cont_e,:) = [element_nodes extra_zeros];
        end
    end

    se = [e_dim e_nodes_num e_surf e_nodes_indexes];


    %% OUTPUT

    data.Mesh.sn = sn;
    data.Mesh.se = se;

    % Reassign old library paths
    setenv('LD_LIBRARY_PATH',MatlabPath)

end
