function [data,st] = ReadPileConfigurationFile(data,fC,execute_mode)

    if exist(fC)~=2
        msg = 'Configuration input file cannot be found';
        f_display_error(execute_mode,msg);
    end

    INP = fopen(fC,'r+');

    PGT.type = lower(fscanf(INP,'%s ',1));
    switch PGT.type
        case 'regular' % PileGroupType data: Regular group
            PGT.bx = fscanf(INP,'%f',1);
            PGT.by = fscanf(INP,'%f',1);
            PGT.Nx = fscanf(INP,'%f',1);
            PGT.Ny = fscanf(INP,'%f',1);
            Ntotal = PGT.Nx*PGT.Ny;

            PGT.sx =   fscanf(INP,'%f',1);
            PGT.sy =   fscanf(INP,'%f',1);
            PGT.l =    fscanf(INP,'%f',1);
            PGT.Rake = fscanf(INP,'%d',1);
            PGT.RakeDirection = fscanf(INP,'%d',1);
            PGT.RakeAngle =     fscanf(INP,'%d',1);

            if PGT.Rake~=1 && PGT.Rake~=2 && PGT.Rake~=3
                msg = 'Not valid Rake';
                f_display_error(execute_mode,msg);
            end
            if PGT.RakeDirection~=1 && PGT.RakeDirection~=2 && PGT.RakeDirection~=3
                msg = 'Not valid Rake Direction';
                f_display_error(execute_mode,msg);
            end
            if PGT.RakeAngle<(-90) || PGT.RakeAngle>(90)
                msg = 'Not valid Rake Angle';
                f_display_error(execute_mode,msg);
            end

            xref = -0.5*(PGT.Nx-1)*PGT.sx;
            yref = -0.5*(PGT.Ny-1)*PGT.sy;
            zref = 0;

            for kpx = 1:PGT.Nx
                for kpy = 1:PGT.Ny
                    % Pile head center
                    xhead(1) = xref + PGT.sx*(kpx-1);
                    xhead(2) = yref + PGT.sy*(kpy-1);
                    xhead(3) = 0;

                    % Pile tip center
                    % Rake direction for the present pile
                    switch PGT.RakeDirection
                        case 1
                            if xhead(1)<0
                                d = [-1 0];
                            elseif xhead(1)==0
                                d = [0 0];
                            else
                                d = [1 0];
                            end
                        case 2
                            if xhead(2)<0
                                d = [0 -1];
                            elseif xhead(2)==0
                                d = [0 0];
                            else
                                d = [0 1];
                            end
                        case 3
                            if sqrt(dot(xhead,xhead))==0
                                d = [0 0];
                            else
                                d = xhead/sqrt(dot(xhead,xhead));
                            end
                    end

                    % Rake angle for the present pile
                    switch PGT.Rake
                        case 1 % no one is inclined
                            theta = 0;
                        case 2 % all are inclined
                            theta = PGT.RakeAngle;
                        case 3 % only outer
                            if kpx==1 || kpx==PGT.Nx || kpy==1 || kpy==PGT.Ny
                                theta = PGT.RakeAngle;
                            else
                                theta = 0;
                            end
                    end

                    Lxp = PGT.l*sind(theta);
                    xtip = xhead + [d(1)*Lxp d(2)*Lxp -PGT.l*cosd(theta)];

                    % Save pile pile and tip center coordinates
                    PGT.PileHeadX(PGT.Nx*(kpy-1)+kpx,:) = xhead;
                    PGT.PileTipX(PGT.Nx*(kpy-1)+kpx,:) = xtip;
                end
            end

            PGT.Do =  fscanf(INP,'%f',1);
            PGT.Di =  fscanf(INP,'%f',1);
            PGT.E =   fscanf(INP,'%f',1);
            PGT.rho = fscanf(INP,'%f',1);

        case 'generic'
            PGT.bx = fscanf(INP,'%f',1);
            PGT.by = fscanf(INP,'%f',1);
            Ntotal = fscanf(INP,'%d',1);

            pile_prop_input_option = lower(fscanf(INP,'%s',1));
            switch pile_prop_input_option
                case 'rake'
                    % % 1 xtop ytop ztop rakeAngle rakeDirection
                    for idpil = 1:Ntotal
                        pil = fscanf(INP,'%d',1);
                        for ax = 1:3
                            PGT.PileHeadX(pil,ax) = fscanf(INP,'%f',1);
                        end
                        PGT.Rake =          fscanf(INP,'%d',1);
                        PGT.RakeDirection = fscanf(INP,'%d',1);
                        PGT.RakeAngle =     fscanf(INP,'%f',1);

                        pil = fscanf(INP,'%d',1);
                        PGT.Do(pil) =  fscanf(INP,'%f',1);
                        PGT.Di(pil) =  fscanf(INP,'%f',1);
                        PGT.E(pil) =   fscanf(INP,'%f',1);
                        PGT.rho(pil) = fscanf(INP,'%f',1);
                    end

                case 'coordinates'
                    % % 1 xtop ytop ztop xtip ytip ztip
                    for idpil = 1:Ntotal
                        for ax = 1:3
                            pil = fscanf(INP,'%d',1);
                            PGT.PileHeadX(pil,ax) = fscanf(INP,'%f',1);
                            PGT.PileTipX(pil,ax)  = fscanf(INP,'%f',1);
                        end

                        pil = fscanf(INP,'%d',1);
                        PGT.Do(pil) =  fscanf(INP,'%f',1);
                        PGT.Di(pil) =  fscanf(INP,'%f',1);
                        PGT.E(pil) =   fscanf(INP,'%f',1);
                        PGT.rho(pil) = fscanf(INP,'%f',1);
                    end

                otherwise
                    msg = 'Incorrect pile properties input option';
                    f_display_error(execute_mode,msg);
            end

%        case 2 % Other groups
            %PGT.Symmetry = 4;
            %PGT.sym_axis = 0;

        otherwise
            msg = 'Problem configuration requested is not implemented yet';
            f_display_error(execute_mode,msg);
    end

    % Translation to general discretization structure of the pile group
    NPiles = 0; % Number of piles in the discretization

    switch PGT.type
        case 'regular'
            for kp = 1:Ntotal
                xh = PGT.PileHeadX(kp,:);
                xt = PGT.PileTipX(kp,:);
                if xh(1)>=0 && xh(2)>=0
                    NPiles = NPiles + 1;
                    PilesHeadX(NPiles,:) = xh;
                    PilesTipX(NPiles,:) = xt;
                end
            end
            Do =  PGT.Do*ones(NPiles,1);
            Di =  PGT.Di*ones(NPiles,1);
            A =   pi*((PGT.Do)^2-(PGT.Di)^2)/4*ones(NPiles,1);
            I =   pi*((PGT.Do)^4-(PGT.Di)^4)/64*ones(NPiles,1);
            E =   PGT.E*ones(NPiles,1);
            rho = PGT.rho*ones(NPiles,1);

        case 'generic'
            for kp = 1:Ntotal
                xh = PGT.PileHeadX(kp,:);
                xt = PGT.PileTipX(kp,:);
                if xh(1)>=0 && xh(2)>=0
                    NPiles = NPiles + 1;
                    PilesHeadX(NPiles,:) = xh;
                    PilesTipX(NPiles,:) = xt;
                    Do(NPiles) = PGT.Do(kp);
                    Di(NPiles) = PGT.Di(kp);
                    A(NPiles) = pi*((PGT.Do(kp))^2-(PGT.Di(kp))^2)/4;
                    I(NPiles) = pi*((PGT.Do(kp))^4-(PGT.Di(kp))^4)/64;
                    E(NPiles) = PGT.E(kp);
                    rho(NPiles) = PGT.rho(kp);
                end
            end
    end


    % Generate general data structure
    Pile = struct('xhead',[],'xtip',[],'Do',[],'Di', [],...
                  'A',    [],'I',   [],'E', [],'rho',[]);

    % Symmetry: 1 (no symmetry), 2 (half-symmetry), 4 (quarter-symmetry)
    PG = struct('Symmetry',4,     ... 
                'N',       NPiles,...
                'Pile',    Pile);

    PG.Pile(NPiles) = Pile; 
    for kp = 1:NPiles
        PG.Pile(kp).xhead = PilesHeadX(kp,:);
        PG.Pile(kp).xtip  = PilesTipX(kp,:);
        PG.Pile(kp).Do    = Do(kp);
        PG.Pile(kp).Di    = Di(kp);
        PG.Pile(kp).A     = A(kp);
        PG.Pile(kp).I     = I(kp);
        PG.Pile(kp).E     = E(kp);
        PG.Pile(kp).rho   = rho(kp);
    end
    
    st = fclose(INP);

    data.PileGroupType = PGT;
    data.PileGroup = PG;
end

function f_display_error(execute_mode,msg)
    switch execute_mode
        case 1 % GUI
            errordlg(msg, 'Alert', 'modal')
        case 2 % command 
            error(msg)
        otherwise
            error('Wrong execution mode selected.')
    end
    return
end
