function data = ReadInputDataFiles(CaseName,Path,fC,fS,fW,fMO,execute_mode)

    %% Configuration data
    if exist(fC)~=2
        msg = 'Configuration input file cannot be found';
        f_display_error(execute_mode,msg);
    end

    INP = fopen(fC,'r+');

    PGT.type = lower(fscanf(INP,'%s ',1));
    switch PGT.type
        case 'regular' % PileGroupType data: Regular group
            PGT.bx = fscanf(INP,'%f',1);
            PGT.by = fscanf(INP,'%f',1);
            PGT.Nx = fscanf(INP,'%f',1);
            PGT.Ny = fscanf(INP,'%f',1);
            Ntotal = PGT.Nx*PGT.Ny;

            PGT.sx =   fscanf(INP,'%f',1);
            PGT.sy =   fscanf(INP,'%f',1);
            PGT.l =    fscanf(INP,'%f',1);
            PGT.Rake = fscanf(INP,'%d',1);
            PGT.RakeDirection = fscanf(INP,'%d',1);
            PGT.RakeAngle =     fscanf(INP,'%d',1);

            if PGT.Rake~=1 && PGT.Rake~=2 && PGT.Rake~=3
                msg = 'Not valid Rake';
                f_display_error(execute_mode,msg);
            end
            if PGT.RakeDirection~=1 && PGT.RakeDirection~=2 && PGT.RakeDirection~=3
                msg = 'Not valid Rake Direction';
                f_display_error(execute_mode,msg);
            end
            if PGT.RakeAngle<(-90) || PGT.RakeAngle>(90)
                msg = 'Not valid Rake Angle';
                f_display_error(execute_mode,msg);
            end

            xref = -0.5*(PGT.Nx-1)*PGT.sx;
            yref = -0.5*(PGT.Ny-1)*PGT.sy;
            zref = 0;

            for kpx = 1:PGT.Nx
                for kpy = 1:PGT.Ny
                    % Pile head center
                    xhead(1) = xref + PGT.sx*(kpx-1);
                    xhead(2) = yref + PGT.sy*(kpy-1);
                    xhead(3) = 0;

                    % Pile tip center
                    % Rake direction for the present pile
                    switch PGT.RakeDirection
                        case 1
                            if xhead(1)<0
                                d = [-1 0];
                            elseif xhead(1)==0
                                d = [0 0];
                            else
                                d = [1 0];
                            end
                        case 2
                            if xhead(2)<0
                                d = [0 -1];
                            elseif xhead(2)==0
                                d = [0 0];
                            else
                                d = [0 1];
                            end
                        case 3
                            if sqrt(dot(xhead,xhead))==0
                                d = [0 0];
                            else
                                d = xhead/sqrt(dot(xhead,xhead));
                            end
                    end

                    % Rake angle for the present pile
                    switch PGT.Rake
                        case 1 % no one is inclined
                            theta = 0;
                        case 2 % all are inclined
                            theta = PGT.RakeAngle;
                        case 3 % only outer
                            if kpx==1 || kpx==PGT.Nx || kpy==1 || kpy==PGT.Ny
                                theta = PGT.RakeAngle;
                            else
                                theta = 0;
                            end
                    end

                    Lxp = PGT.l*sind(theta);
                    xtip = xhead + [d(1)*Lxp d(2)*Lxp -PGT.l*cosd(theta)];

                    % Save pile pile and tip center coordinates
                    PGT.PileHeadX(PGT.Nx*(kpy-1)+kpx,:) = xhead;
                    PGT.PileTipX(PGT.Nx*(kpy-1)+kpx,:) = xtip;
                end
            end

            PGT.Do =  fscanf(INP,'%f',1);
            PGT.Di =  fscanf(INP,'%f',1);
            PGT.E =   fscanf(INP,'%f',1);
            PGT.rho = fscanf(INP,'%f',1);

            if size(PGT.Do,1)==1
                PGT.Do = PGT.Do*ones(Ntotal,1);
            end
            if size(PGT.Di,1)==1
                PGT.Di = PGT.Di*ones(Ntotal,1);
            end
            if size(PGT.E,1)==1
                PGT.E = PGT.E*ones(Ntotal,1);
            end
            if size(PGT.rho,1)==1
                PGT.rho = PGT.rho*ones(Ntotal,1);
            end

        case 'generic'
            PGT.bx = fscanf(INP,'%f',1);
            PGT.by = fscanf(INP,'%f',1);
            Ntotal = fscanf(INP,'%d',1);

            pile_prop_input_option = lower(fscanf(INP,'%s',1));
            switch pile_prop_input_option
                case 'rake'
                    % % 1 xtop ytop ztop rakeAngle rakeDirection
                    for idpil = 1:Ntotal
                        pil = fscanf(INP,'%d',1);
                        for ax = 1:3
                            PGT.PileHeadX(pil,ax) = fscanf(INP,'%f',1);
                        end
                        PGT.Rake =          fscanf(INP,'%d',1);
                        PGT.RakeDirection = fscanf(INP,'%d',1);
                        PGT.RakeAngle =     fscanf(INP,'%f',1);

                        pil = fscanf(INP,'%d',1);
                        PGT.Do(pil) =  fscanf(INP,'%f',1);
                        PGT.Di(pil) =  fscanf(INP,'%f',1);
                        PGT.E(pil) =   fscanf(INP,'%f',1);
                        PGT.rho(pil) = fscanf(INP,'%f',1);
                    end

                case 'coordinates'
                    % % 1 xtop ytop ztop xtip ytip ztip
                    for idpil = 1:Ntotal
                        for ax = 1:3
                            pil = fscanf(INP,'%d',1);
                            PGT.PileHeadX(pil,ax) = fscanf(INP,'%f',1);
                            PGT.PileTipX(pil,ax)  = fscanf(INP,'%f',1);
                        end

                        pil = fscanf(INP,'%d',1);
                        PGT.Do(pil) =  fscanf(INP,'%f',1);
                        PGT.Di(pil) =  fscanf(INP,'%f',1);
                        PGT.E(pil) =   fscanf(INP,'%f',1);
                        PGT.rho(pil) = fscanf(INP,'%f',1);
                    end

                otherwise
                    msg = 'Incorrect pile properties input option';
                    f_display_error(execute_mode,msg);
            end

%        case 2 % Other groups
            %PGT.Symmetry = 4;
            %PGT.sym_axis = 0;

        otherwise
            msg = 'Problem configuration requested is not implemented yet';
            f_display_error(execute_mode,msg);
    end

    % Translation to general discretization structure of the pile group
    NPiles = 0; % Number of piles in the discretization
    for kp = 1:Ntotal
        xh = PGT.PileHeadX(kp,:);
        xt = PGT.PileTipX(kp,:);
        if xh(1)>=0 && xh(2)>=0
            NPiles = NPiles + 1;
            PilesHeadX(NPiles,:) = xh;
            PilesTipX(NPiles,:) = xt;
            Do(NPiles) = PGT.Do(kp);
            Di(NPiles) = PGT.Di(kp);
            A(NPiles) = pi*((PGT.Do(kp))^2-(PGT.Di(kp))^2)/4;
            I(NPiles) = pi*((PGT.Do(kp))^4-(PGT.Di(kp))^4)/64;
            E(NPiles) = PGT.E(kp);
            rho(NPiles) = PGT.rho(kp);
        end
    end

    % Generate general data structure
    Pile = struct('xhead',[],'xtip',[],'Do',[],'Di', [],...
                  'A',    [],'I',   [],'E', [],'rho',[]);

    % Symmetry: 1 (no symmetry), 2 (half-symmetry), 4 (quarter-symmetry)
    PG = struct('Symmetry',4,     ... 
                'N',       NPiles,...
                'Pile',    Pile);

    PG.Pile(NPiles) = Pile; 
    for kp = 1:NPiles
        PG.Pile(kp).xhead = PilesHeadX(kp,:);
        PG.Pile(kp).xtip  = PilesTipX(kp,:);
        PG.Pile(kp).Do    = Do(kp);
        PG.Pile(kp).Di    = Di(kp);
        PG.Pile(kp).A     = A(kp);
        PG.Pile(kp).I     = I(kp);
        PG.Pile(kp).E     = E(kp);
        PG.Pile(kp).rho   = rho(kp);
    end
    
    fclose(INP);
    PileGroupType = PGT;
    PileGroup = PG;


    %% Stratigraphy
    if exist(fS)~=2
        msg = 'SoilStratigraphy input file cannot be found';
        f_display_error(execute_mode,msg);
    end
    INP = fopen(fS,'r+');


    Soil.N = fscanf(INP,'%d',1);         % Number of strata

    name = cell(Soil.N,1); % Name of each stratum
    zdom = zeros(Soil.N,1); % Higher z coordinate of each stratum
    mus =  zeros(Soil.N,1); % Shear modulus of elasticity
    nus =  zeros(Soil.N,1); % Poisson’s ratio
    rhos = zeros(Soil.N,1); % Mass density
    xis =  zeros(Soil.N,1); % Damping ratio
    bedrock = zeros(Soil.N,1);

    for idstratum = 1:Soil.N
        name{idstratum} =    fscanf(INP,'%s',1);
        zdom(idstratum) =    fscanf(INP,'%f',1);
        bedrock(idstratum) = fscanf(INP,'%d',1);

        switch bedrock(idstratum)
            case 0
                mus(idstratum) =  fscanf(INP,'%f',1);
                nus(idstratum) =  fscanf(INP,'%f',1);
                rhos(idstratum) = fscanf(INP,'%f',1);
                xis(idstratum) =  fscanf(INP,'%f',1);
            case 1
                % nothing
            otherwise
                msg = 'Incorrect bedrock option';
                f_display_error(execute_mode,msg);
        end
    end

    if isempty(Soil.N) || isempty(name) || isempty(mus) || isempty(nus) || ...
         isempty(rhos) || isempty(xis) || isempty(zdom)
        msg = 'Soil input incomplete';
        f_display_error(execute_mode,msg);
    end
    if ( (length(mus)~=Soil.N) || (length(rhos)~=Soil.N) || (length(nus)~=Soil.N) || ...
         (length(xis)~=Soil.N) || (length(zdom)~=Soil.N)                             )
        msg = 'Soil properties are not defined properly';
        f_display_error(execute_mode,msg);
    end

    zdom = (-1)*abs(zdom);

    % missing function: check order strata and properties

    switch max(bedrock)
        case 1
            Soil.bedrock = true;
        case 0
            Soil.bedrock = false;
    end

    for iddom = 1:Soil.N
        if iddom~=Soil.N || (iddom==Soil.N && ~Soil.bedrock)
            Soil.layer(iddom).mu =  mus(iddom);
            Soil.layer(iddom).rho = rhos(iddom);
            Soil.layer(iddom).nu =  nus(iddom);
            Soil.layer(iddom).xi =  xis(iddom);
            Soil.layer(iddom).cs =  sqrt(mus(iddom)/rhos(iddom));
        else
            Soil.layer(iddom).cs = Inf;
        end
        Soil.layer(iddom).name = name{iddom};
        Soil.layer(iddom).z =    zdom(iddom);
    end

    fclose(INP);


    %% Frequencies
    if exist(fW)~=2
        msg = 'Frequencies input file cannot be found';
        f_display_error(execute_mode,msg);
    end
    INP= fopen(fW,'r+');

    W.type = fscanf(INP,'%d',1);
    nfreq =  fscanf(INP,'%d',1);

    W.value = zeros(nfreq,1);
    for idfreq = 1:nfreq
        W.value(idfreq) = fscanf(INP,'%f',1); % Vector of values
    end
    
    W.ref = fscanf(INP,'%d',1); % 1) D,c , 2) w_ref, 3) f_ref
    if W.ref~=1 && W.ref~=2 && W.ref~=3
        msg = 'Incorrect frequencie reference type';
        f_display_error(execute_mode,msg);
    end
    switch W.ref
        case 1
            W.ref_data(1) = fscanf(INP,'%f',1);
            W.ref_data(2) = fscanf(INP,'%f',1);
        case {2,3}
            W.ref_data = fscanf(INP,'%f',1);
    end

    fclose(INP);

    if isempty(W.type) || isempty(W.value) || isempty(W.ref) || isempty(W.ref_data)
        msg = 'Frequencie input incomplete';
        f_display_error(execute_mode,msg);
    end
    if W.type~=1 && W.type~=2 && W.type~=3
        msg = 'Incorrect frequencie type';
        f_display_error(execute_mode,msg);
    end
    if (W.ref==1 && length(W.ref_data)~=2) || (W.ref==2 && length(W.ref_data)~=1)
        msg = 'Incorrect frequencie reference value(s)';
        f_display_error(execute_mode,msg);
    end
    Frequencies = W;


    %% Mesh Options
    if exist(fMO)==2
        % File does exist
        INP = fopen(fMO,'r+');
        tool =      fscanf(INP,'%d',1);
        ratio_r =   fscanf(INP,'%f',1);
        ratio_R =   fscanf(INP,'%f',1);
        ratio_D =   fscanf(INP,'%f',1);
        nepl_near = fscanf(INP,'%f',1);
        nepl_far =  fscanf(INP,'%f',1);
        nepd_pile = fscanf(INP,'%f',1);
        nelo_near = fscanf(INP,'%f',1);
        nelo_far =  fscanf(INP,'%f',1);
        nelo_pile = fscanf(INP,'%f',1);
        t =         fscanf(INP,'%f',1);
        tolerance = fscanf(INP,'%f',1);
        fclose(INP);

        if isempty(ratio_R)    || isempty(ratio_r)   || ...
            isempty(ratio_D)   || isempty(tool)      || ...
            isempty(nepl_near) || isempty(nelo_near) || ...
            isempty(nepl_far)  || isempty(nelo_far)  || ...
            isempty(nepd_pile) || isempty(nelo_pile) || ...
            isempty(tolerance) || isempty(t)

            msg = 'Mesh options input incomplete';
            f_display_error(execute_mode,msg);
        end

        switch tool
            case 1;     toolpath = strcat(pwd,'/mesh2d/');
            case 2;     toolpath = 'gmsh';
        end

        Mopt = struct('tool',tool,'toolpath', toolpath,'t',t,     ...
                      'tolerance',tolerance,'ratio_R',  ratio_R,  ...
                      'ratio_r',  ratio_r,  'ratio_D',  ratio_D,  ...
                      'nepl_near',nepl_near,'nelo_near',nelo_near,...
                      'nepl_far', nepl_far, 'nelo_far', nelo_far, ...
                      'nepd_pile',nepd_pile,'nelo_pile',nelo_pile); 
    else
        % File does not exist
        Mopt = struct('tool',1,'toolpath',strcat(pwd,'/mesh2d/'),    ...
                      'ratio_r',1.5,'ratio_D',3,'ratio_R',3,         ...
                      'nepl_near',1,'nelo_near',6,'nepl_far',4,      ...
                      'nelo_far',2,'nepd_pile',1,'nelo_pile',2,'t',0,...
                      'tolerance',1.e-6); 
    end
    MeshOptions = Mopt;


    %% Analysis
    % Impedances: (1) x , (2) y , (3) z , (4) rx , (5) ry
    Impedances = ones(1,5);

    if isempty(CaseName)
        % Proyect Name may be introduced by user
        prompt = 'Enter the project name: ';
        switch execute_mode
            case 1 % GUI
                dlg_title = 'Project Name';
                CaseName = char(inputdlg(prompt,dlg_title,[1 50]));
            case 2 % command
                CaseName = input(prompt,'s');
            otherwise
                error('Wrong execution mode selected.')
        end
    end

    if ~exist(CaseName,'dir')
        msg = strcat('''',CaseName,''' folder cannot be found');
        f_display_error(execute_mode,msg);
    end

    Analysis = struct('Impedances',Impedances,   ...
                      'Name',      CaseName   );


    %% OUTPUT
    data = struct('Name',         CaseName,      ...
                  'Path',         Path,          ...
                  'Soil',         Soil,          ...
                  'PileGroupType',PileGroupType, ...
                  'PileGroup',    PileGroup,     ...
                  'Frequencies',  Frequencies,   ...
                  'MeshOptions',  MeshOptions,   ...
                  'Mesh',         [],            ...
                  'Analysis',     Analysis);
end


function f_display_error(execute_mode,msg)
    switch execute_mode
        case 1 % GUI
            errordlg(msg, 'Alert', 'modal')
        case 2 % command 
            error(msg)
        otherwise
            error('Wrong execution mode selected.')
    end
    return
end
