function st = OUT_BEMFEMfile(filename,imp,data)

    S = data.Soil;
    F = data.Frequencies;
    P = data.PileGroup;
    M = data.Mesh;
    
    if S.bedrock
        NDomains = S.N-1;
    else
        NDomains = S.N;
    end
    NBoundaries = S.N;
    
    OUP = fopen(filename,'w+');

    %% Frequencies
    switch F.ref
        case 1
            wref = F.ref_data(2)/F.ref_data(1);
        case 2
            wref = F.ref_data(1);
        case 3
            wref = F.ref_data(1)*2*pi;
    end
    switch F.type
        case 1
            a0 = F.value;
        case 2
            a0 = F.value/wref;
        case 3 
            a0 = F.value/(wref*2*pi);
    end
    str=sprintf('%d %d %23.16E',NDomains,length(a0),wref);
    str=strrep(str,'E','D');
    fprintf(OUP,'%s\n\n',str);
    for k=1:length(a0)
        fprintf(OUP,'%s\n',strrep(sprintf('%23.16E',a0(k)),'E','D'));
    end
    fprintf(OUP,'\n');

    %% Domains
    
    % Type of domain
    for k=1:NDomains
        fprintf(OUP,'%d 2\n',k);
    end
    fprintf(OUP,'\n');
    % Properties of each domain
    for k=1:NDomains
        fprintf(OUP,'%d 0\n',k);
        str = sprintf('%d %23.16E %23.16E %23.16E %23.16E 0',...
            k,S.layer(k).mu,S.layer(k).nu,S.layer(k).xi,S.layer(k).rho);
        str = strrep(str,'E','D');
        fprintf(OUP,'%s\n\n',str);
    end
    

    %% Mesh

    % Nodes
    fprintf(OUP,'%d %d\n\n',size(M.sn,1),size(M.sn,1)+size(M.pn,1));
    for k=1:size(M.sn,1)
        str = sprintf('%d %23.16E %23.16E %23.16E',k,M.sn(k,:));
        str = strrep(str,'E','D');
        fprintf(OUP,'%s\n',str);
        
    end
    fprintf(OUP,'\n');

    % Elements
    fprintf(OUP,'%d %d\n\n',size(M.se,1),size(M.se,1));
    for k=1:size(M.se,1)
        fprintf(OUP,'%d %d ',k,M.se(k,2));
        switch M.se(k,2)
            case 6
                for n=[1 2 3 4 5 6]
                    fprintf(OUP,'%d ',M.se(k,3+n));
                end
            case 9
                for n=[1 5 2 6 3 7 4 8 9]
                    fprintf(OUP,'%d ',M.se(k,3+n));
                end
            otherwise
                error('Invalid element type')
        end
        fprintf(OUP,'\n');
    end
    fprintf(OUP,'\n');

    % Number of boundaries
    if S.N==1
        fprintf(OUP,'%d %d 1\n',S.N,S.N);
    else
        fprintf(OUP,'%d %d 2\n',S.N,S.N);
    end
    fprintf(OUP,'\n');

    % Number of elements of each boundary
    for k=1:S.N
        e1 = find(M.se(:,3)==2*k-1); 
        e2 = find(M.se(:,3)==2*k); 
        ne1 = length(e1);
        ne2 = length(e2);
        fprintf(OUP,'%d %d %d %d\n',k,ne1+ne2,e1(1),e1(ne1)); 
    end
    fprintf(OUP,'\n');

    % Boundaries of each domain 
    if S.bedrock
        for k=1:NDomains
            fprintf(OUP,'%d 2 %d %d\n',k,k,-(k+1));
        end
    else
        for k=1:NDomains-1
            fprintf(OUP,'%d 2 %d %d\n',k,k,-(k+1));
        end
        fprintf(OUP,'%d 1 %d\n',NDomains,NBoundaries);
    end
    fprintf(OUP,'\n');

    % Boundary conditions for each domain 
    fprintf(OUP,'1 22\n');
    if S.bedrock
        for k=2:NBoundaries-1
            fprintf(OUP,'%d 7\n',k);
        end
        fprintf(OUP,'%d 32\n',NBoundaries);
    else
        for k=2:NBoundaries
            fprintf(OUP,'%d 7\n',k);
        end
    end
    fprintf(OUP,'\n');

    % Symmetry conditions
    % imp: 1(ux), 2(uy), 3(uz), 4(thetax), 5(thetay)
    switch imp
        case 1
            % VX VY VZ S
            fprintf(OUP,' 1 -1  1  1\n'); % XZ (SYMMETRY)
            fprintf(OUP,' 1 -1 -1 -1\n'); % YZ (ANTISYMMETRY)
        case 2
            % VX VY VZ S
            fprintf(OUP,'-1  1 -1 -1\n'); % XZ (ANTISYMMETRY)
            fprintf(OUP,'-1  1  1  1\n'); % YZ (SYMMETRY)
        case 3
            % VX VY VZ S
            fprintf(OUP,' 1 -1  1  1\n'); % XZ (SYMMETRY)
            fprintf(OUP,'-1  1  1  1\n'); % YZ (SYMMETRY)
        case 4
            % VX VY VZ S
            fprintf(OUP,'-1  1 -1 -1\n'); % XZ (ANTISYMMETRY)
            fprintf(OUP,'-1  1  1  1\n'); % YZ (SYMMETRY)
        case 5
            % VX VY VZ S
            fprintf(OUP,' 1 -1  1  1\n'); % XZ (SYMMETRY)
            fprintf(OUP,' 1 -1 -1 -1\n'); % YZ (ANTISYMMETRY)
    end
    fprintf(OUP,'\n');
    
    % Number of singular nodes
    fprintf(OUP,'0 1\n\n');
    
    % Type of problem
    fprintf(OUP,'0\n\n' );
    
    % Type of line load
    fprintf(OUP,'1\n\n' );
    
    % Number of line loads in each domain
    nlcd  = zeros(1,NDomains);
    nlcid = zeros(1,NDomains);
    nlcfd = zeros(1,NDomains);
    for k=1:NDomains
        el = find(M.pe(:,2)==k);
        if ~isempty(el)
            nlcd(k) = length(unique(M.pe(el,3)));
            nlcid(k) = min(unique(M.pe(el,3)));
            nlcfd(k) = max(unique(M.pe(el,3)));
        end
        fprintf(OUP,'%d %d\n',k,nlcd(k));
    end
    fprintf(OUP,'\n');    
    
    % Definition of line loads per domain
    for k=1:NDomains
        if nlcd(k) == 0
            continue
        end
        fprintf(OUP,'%d %d\n\n',nlcid(k),nlcfd(k));
        el = find(M.pe(:,2)==k);
        lc = unique(M.pe(el,3));
        % Nodes of each line load - pile
        for l=1:nlcd(k)
            elc = find(M.pe(:,3)==lc(l));
            nlc = unique(M.pe(elc,6:8));
            [~,node_bot] = min(M.pn(nlc,3));
            node_bot = nlc(node_bot);
            [~,node_top] = max(M.pn(nlc,3));
            node_top = nlc(node_top);
            % Check if there is no gap in nodes for each line load
            if ~isequal(nlc,[min(nlc):max(nlc)]')
                error('Pile mesh has been incorrectly generated.')
            end
            % Bottom and top nodes
            fprintf(OUP,'%d %d\n',node_bot+length(M.sn(:,1)),node_top+length(M.sn(:,1)));
            % Non-nodal collocation
            if k<NDomains
                if abs(M.pn(node_bot,3)-S.layer(k+1).z)<1.0e-6
                    fprintf(OUP,'0 1 0 1\n');
                else
                    fprintf(OUP,'0 1 0 0\n');
                end
            else
                fprintf(OUP,'0 1 0 0\n');
            end
            % Coordinates of bottom and top nodes
            str = sprintf('%23.16E %23.16E',M.pn(node_bot,1),M.pn(node_top,1));
            str = strrep(str,'E','D');
            fprintf(OUP,'%s\n',str);
            str = sprintf('%23.16E %23.16E',M.pn(node_bot,2),M.pn(node_top,2));
            str = strrep(str,'E','D');
            fprintf(OUP,'%s\n',str);
            str = sprintf('%23.16E %23.16E',M.pn(node_bot,3),M.pn(node_top,3));
            str = strrep(str,'E','D');
            fprintf(OUP,'%s\n\n',str);
        end
        % Pile properties
        for l=1:nlcd(k)
            elc = find(M.pe(:,3)==lc(l));
            pile=M.pe(elc(1),1);
            str=sprintf('%23.16E %23.16E %23.16E %23.16E %23.16E',...
                P.Pile(pile).E,P.Pile(pile).rho-S.layer(k).rho,...
                P.Pile(pile).A,P.Pile(pile).I,P.Pile(pile).Do/2);
            str = strrep(str,'E','D');
            fprintf(OUP,'%s\n',str);
        end
        fprintf(OUP,'\n');
        % Boundary conditions of each line load - pile
        if k==1
            for l=1:nlcd(k)
                elc = find(M.pe(:,3)==lc(l));
                pile=M.pe(elc(1),1);
                fprintf(OUP,'0 1\n');
                % imp: 1(ux), 2(uy), 3(uz), 4(thetax), 5(thetay), 6(thetaz)
                switch imp
                    case 1
                        fprintf(OUP,'0 (1.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                    case 2
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (1.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                    case 3
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (1.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                    case 4
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        str = sprintf('0 (%+23.16E,0.D0)',P.Pile(pile).xhead(2));
                        str = strrep(str,'E','D');
                        fprintf(OUP,'%s\n',str);
                        fprintf(OUP,'0 (0.D0,0.D0)\n'); % DOF: rotation such that ux>0 ( theta_y)
                        fprintf(OUP,'0 (-1.D0,0.D0)\n'); % DOF: rotation such that uy>0 (-theta_x)
                    case 5
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        fprintf(OUP,'0 (0.D0,0.D0)\n');
                        str = sprintf('0 (%+23.16E,0.D0)',-P.Pile(pile).xhead(1));
                        str = strrep(str,'E','D');
                        fprintf(OUP,'%s\n',str);
                        fprintf(OUP,'0 (1.D0,0.D0)\n'); % DOF: rotation such that ux>0 ( theta_y)
                        fprintf(OUP,'0 (0.D0,0.D0)\n'); % DOF: rotation such that uy>0 (-theta_x)
                end
            end
        else
            for l=1:nlcd(k)
                fprintf(OUP,'0 0\n');
            end
        end
        fprintf(OUP,'\n');
    end
    
    % Coupling between line loads
    % Calculate double nodes in pile mesh
    dn = zeros(n,2);
    ndn = 0;
    for n1=1:length(M.pn(:,1))
        kn1 = 0;
        for n2=(n1+1):length(M.pn(:,1))
            r = M.pn(n1,:)-M.pn(n2,:);
            if norm(r)<1.0e-6
                if kn1 == 0
                    ndn = ndn + 1;
                    kn1 = kn1 + 1;
                    dn(ndn,1) = n1;
                    dn(ndn,2) = n2;
                else
                    error ('Pile mesh has been incorrectly generated.')
                end
            end
        end
    end
    dn = dn(1:ndn,:);
    % Write coupling between line loads
    fprintf(OUP,'%d\n',ndn);
    for k=1:ndn
        fprintf(OUP,'%d %d\n',dn(k,1)+length(M.sn(:,1)),dn(k,2)+length(M.sn(:,1)));
    end
    fprintf(OUP,'\n'); 
    
    % Other flags
    for cont=1:20
       fprintf(OUP,'0\n'); 
    end

    st = fclose(OUP);
   
    % fclose returns 0 if successful or -1 if not
    if st==0
        fprintf('File %s creation succeded.\n',filename);
    else
        warning('File %s creation failed.\n',filename);
    end

end
