function data = MainFILE(varargin)
    %% INPUT
    if length(varargin)==0
        disp('Enter the input argument:')
        generate_mesh = logical(input('Generate mesh(1) or load it(0)?\n'));
        view_mesh =     logical(input('View mesh? yes(1)/no(0)\n'));
        if isempty(generate_mesh) || isempty(view_mesh)
            error('Empty argument. Check input data.');
        end

        CaseName = input('Problem name: \n','s');
        Path =     input('Path to directory:\n','s');
        if isempty(Path) || isempty(CaseName)
            error('Empty argument. Check input data.');
        end

        if ~exist(Path,'dir')
            fprintf(strcat('''',CaseName,''' directory cannot be found.\n'));
            Path = input('Enter full path to directory: \n','s');

            [folder_path,folder] = fileparts(Path);
            if folder~=CaseName
                error('Wrong path to directory');
            end
        end

        for id_f = 1:4
            switch id_f
                case 1
                    msg = 'Enter Pile Configuration File name:\n';
                    ext = '.pil';
                case 2
                    msg = 'Enter Soil Stratigraphy File name:\n';
                    ext = '.soi';
                case 3
                    msg = 'Enter Frequencies File name:\n';
                    ext = '.fre';
                case 4
                    if generate_mesh
                        msg = 'Enter Mesh Options File name:\n';
                        ext = '.mop';
                    else
                        msg = 'Enter Mesh File name:\n';
                        ext = '.mes';
                    end
            end

            [file_path,file] = fileparts(input(msg,'s'));
            f = fullfile(Path,strcat(file,ext));

            if isempty(f)
                error('Empty argument. Check input data.');
            end
            if ~exist(f,'file')
                error('Input file cannot be found. Check input data.');
            end

            switch id_f
                case 1
                    fC = f;
                case 2
                    fS = f;
                case 3
                    fW = f;
                case 4
                    if generate_mesh
                        fMO = f;
                        fUserMesh = fullfile(Path,strcat(CaseName,'.mes'));
                    else
                        fMO = ' ';
                        fUserMesh = f;
                    end
            end
        end

    else
        generate_mesh = logical(varargin{1});
        view_mesh =     logical(varargin{2});
        if isempty(generate_mesh) || isempty(view_mesh)
            error('Wrong input arguments');
        end

        CaseName = varargin{3};
        Path =     varargin{4};
        if isempty(CaseName)
            % Proyect Name may be introduced by user
            CaseName = input('Enter the project name: ','s');
        end

        if ~exist(Path,'dir')
            fprintf(strcat('''',CaseName,''' directory cannot be found.\n'));
            Path = input('Enter directory''s full path: ','s');

            [folder_path,folder] = fileparts(Path);
            if folder~=CaseName
                error('Wrong path to directory');
            end
        end

        if length(varargin)==4
            fC = fullfile(Path,strcat(CaseName,'.pil'));
            fS = fullfile(Path,strcat(CaseName,'.soi'));
            fW = fullfile(Path,strcat(CaseName,'.fre'));
            fMO = fullfile(Path,strcat(CaseName,'.mop'));
            fUserMesh = fullfile(Path,strcat(CaseName,'.mes'));
        elseif length(varargin)>5 && length(varargin)<=8
            fC = fullfile(Path,varargin{5});
            fS = fullfile(Path,varargin{6});
            fW = fullfile(Path,varargin{7});
            if generate_mesh
                fMO = fullfile(Path,varargin{8});
                fUserMesh = fullfile(Path,strcat(CaseName,'.mes'));
            else
                fMO = '';
                fUserMesh = fullfile(Path,varargin{8});
            end
        else
            error('Wrong input arguments');
        end
    end


    data = struct('Name',CaseName,...
              'Path',Path,...
              'Soil',[],...
              'PileGroupType',[],...
              'PileGroup',[],...
              'Frequencies',[],...
              'MeshOptions',[],...
              'Mesh',[],...
              'Analysis',struct('VibrationModes',ones(1,6)));

    data = ReadPileConfigurationFile(data,fC,2);
    data = ReadSoilStratigraphyFile(data,fS,2);
    data = ReadFrequenciesFile(data,fW,2);


    %% MAIN
    if generate_mesh
        data = ReadMeshOptionsFile(data,fMO,2);

        % Creating mesh
        switch data.MeshOptions.tool
            case 1;     data = msh_MESH2D(data);
            case 2;     data = msh_Gmsh_S4(data);
            otherwise
                error('Invalid meshing tool');
        end
        % Write Mesh file
        WriteMeshFile(data,fUserMesh);
    else
        data.MeshOptions = struct('nepd_pile',1,'nelo_pile',2,...
                                  'tolerance',1e-6,'t',0);
        
        % Loading mesh specified by user
        data = msh_UserMesh(data,2,fUserMesh);
    end

    % Generating pile mesh
    data = msh_piles(data,2);

    % Check mesh properties are all ok
    data = CheckMesh(data);

    % VIEW MESH
    if view_mesh
        % Surface mesh
        nx  = data.Mesh.sn;
        en  = data.Mesh.se(:,4:12);
        enn = data.Mesh.se(:,2);
        ed  = data.Mesh.se(:,1);
        % Pile mesh
        nxp  = data.Mesh.pn;
        enp  = data.Mesh.pe(:,6:14);
        ennp = data.Mesh.pe(:,5);
        edp  = data.Mesh.pe(:,4);
        % Plot
        figure('Name',CaseName,'NumberTitle','off');
        ax = gca;
        PlotMesh(ax,[nx;nxp],[en;enp+length(nx(:,1))],[enn;ennp],[ed;edp],true);
    end

    %% OUTPUT: GENERATION OF THE PILOT_SIMPL INPUT FILE
    st = ones(1,5);
    for vibr = 1:5
        if data.Analysis.VibrationModes(vibr)>0
            switch vibr
                case 1; fname_end = '_x';
                case 2; fname_end = '_y';
                case 3; fname_end = '_z';
                case 4; fname_end = '_rx';
                case 5; fname_end = '_ry';
            end

            filename = fullfile(data.Path,strcat(data.Name,fname_end,'.dat'));
            st(vibr) =  OUT_BEMFEMfile(filename,vibr,data);
        end
    end

    % st returns 0 if successful or -1 if not
    if min(st) == 0
        disp('File creation succeded ...');
    elseif min(st) == -1
        disp('File creation failed.');
    end

    % Calculate approximate required RAM memory
    memRAM = CalculateRAM(data.Mesh.sn(:,3),length(data.Mesh.pn(:,1)));
    str = sprintf('Approximate required memory is %g GB',memRAM);
    fprintf('%s ...\n',str);
end
