function  data = CheckMesh(data)
        
    % INPUT
    sn = data.Mesh.sn;
    se = data.Mesh.se;
    pn = data.Mesh.pn;
    pe = data.Mesh.pe;
    S =  data.Soil;
    C =  data.PileGroup;
    tol=abs(data.MeshOptions.tolerance);
    
    % Guarantee a 0 to remaining columns in se and pe
    for k=1:length(se(:,1))
        for j=4+se(k,2):length(se(1,:))
            se(k,j) = 0;
        end
    end
    for k=1:length(pe(:,1))
        for j=6+pe(k,5):length(pe(1,:))
            pe(k,j) = 0;
        end
    end

    % Check if mesh coordinates along z axis are all negative
    if ~isempty(find(sn(:,3)>tol)) || ~isempty(find(pn(:,3)>tol))
        error('All nodes must be at z<=0.');
        status = -1;
    end

    % Check if mesh coordinates in x-y plane are all in the considered 
    % region depending on the symmetry.
    switch C.Symmetry
        % No symmetry
        case 1
        % 1/2 symmetry
        case 2
            if ~isempty(find(sn(:,C.Axis)<-tol))
                if C.Axis==1
                    str = 'x';
                else
                    str = 'y';
                end
                error('All nodes must be at %s>=0.',str);
            else
                nn = find(abs(sn(:,C.Axis))<tol);
                sn(nn,C.Axis)=0;
            end
        % 1/4 symmetry    
        case 4
            if ~isempty(find(sn(:,1)<-tol)) || ~isempty(find(sn(:,2)<-tol))
                error('All nodes must be at x>=0 and y>=0.');
            else
                nn = find(abs(sn(:,1))<tol);
                sn(nn,1)=0;
                nn = find(abs(sn(:,2))<tol);
                sn(nn,2)=0;
            end
    end
    
    % Required surfaces
    surfaces = unique(se(:,3))';
    if ~isequal(surfaces,1:2*S.N)
        error('The required surfaces are not present in the mesh.');
    end

    % Check z coordinate of the nodes of each stratum, and pile-stratum
    % intersection
    for k = 1:S.N
        z = S.layer(k).z;
        % Surfaces of the stratum
        s1 = 2*k-1;
        s2 = 2*k;
        % Nodes of the stratum
        n1 = unique(se(se(:,3)==s1,4:12));
        n1 = n1(n1>0);
        n2 = unique(se(se(:,3)==s2,4:12));
        n2 = n2(n2>0);
        nn = union(n1,n2);
        if ~isempty(find(sn(nn,3)>(z+tol))) || ~isempty(find(sn(nn,3)<(z-tol)))
            error('Layer %d (surf. %d and %d) has nodes with invalid z.',k,s1,s2);
        else
            sn(nn,3) = z;
        end
        % Intersection of piles and stratum
        pnn = find(abs(pn(:,3)-z)<tol);
        for l=1:length(pnn)
            r = sqrt((sn(nn,1)-pn(pnn(l),1)).^2+(sn(nn,2)-pn(pnn(l),2)).^2);
            pnnn = find(r<tol);
            if length(pnnn)~=1
                error('Layer %d (surf. %d and %d) must have a node at (%g,%g,%g).',k,s1,s2,pn(pnn(l),:));
            else
                sn(nn(pnnn),1) = pn(pnn(l),1);
                sn(nn(pnnn),2) = pn(pnn(l),2);
            end
        end
    end

    % Check element orientation
    CheckElementNormal = true(1,size(se,1));
    tri = zeros(2,3);
    for k = 1:size(se,1)
        switch se(k,2)
            case 6
                ntri = 1;
                tri(1,1) = se(k,4);
                tri(1,2) = se(k,5);
                tri(1,3) = se(k,6);
            case 9
                ntri = 2;
                tri(1,1) = se(k,4);
                tri(1,2) = se(k,5);
                tri(1,3) = se(k,6);
                tri(2,1) = se(k,4);
                tri(2,2) = se(k,6);
                tri(2,3) = se(k,7);
        end
        for ktri=1:ntri
            r12 = sn(tri(ktri,2),:)-sn(tri(ktri,1),:); 
            r13 = sn(tri(ktri,3),:)-sn(tri(ktri,1),:);
            n = cross(r12,r13);
            n = n/norm(n);
            if sqrt((n(1))^2+(n(2))^2+(n(3)-1)^2)>tol
                warning('Invalid normal: element %d, surface %d.',k,se(k,3));
                CheckElementNormal(k) = false;
            end
        end
    end
    if ~isempty(find(CheckElementNormal==false))
       error('There are elements with invalid orientation (see above).');
    end

    % Save changes
    data.Mesh.sn = sn;
    data.Mesh.se = se;
    data.Mesh.pn = pn;
    data.Mesh.pe = pe;
    
end


